package easik.ui.datatype;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;

import easik.Easik;
import easik.sketch.datatype.DataType;
import easik.sketch.datatype.DataTypeController;


/**
 * This class is used to change information related to the data types
 * available for use in the current document.
 * 
 * @author Kevin Green 2006
 * @since 2006-06-21 Kevin Green
 * @version 2006-06-27 Kevin Green
 */
public class DatatypesUI extends JFrame {

	/**
	 * The starting length size of the window
	 */
	private static final int BASE_WINDOW_LENGTH = 250;
	/**
	 * The starting height size of the window
	 */
	private static final int BASE_WINDOW_HEIGHT = 150;
	/**
	 * The extra length to add on per data type platform
	 */
	private static final int TYPE_WINDOW_LENGTH = 170;
	/**
	 * The extra height to add on per data type
	 */
	private static final int TYPE_WINDOW_HEIGHT = 75;
	/**
	 * The maximum window length size
	 */
	private static final int MAX_WINDOW_LENGTH = 1024;
	/**
	 * The maximum window height size
	 */
	private static final int MAX_WINDOW_HEIGHT = 768;
	/**
	 * The starting X position of the window
	 */
	private static final int WINDOW_POS_X = 0;
	/**
	 * The starting Y position of the window
	 */
	private static final int WINDOW_POS_Y = 0;
	
	/**
	 * A self reference to be used by inner classes
	 */
	private JFrame _me;
	/**
	 * The main pane
	 */
	private JPanel _mainPane;
	/**
	 * The labels pane
	 */
	private JPanel _labelPane;
	/**
	 * The extra pane used to push over the labels
	 */
	private JPanel _labelAdjusterPane;
	/**
	 * The data types pane
	 */
	private JPanel _dataPane;
	/**
	 * The bottom button pane
	 */
	private JPanel _buttonPane;
	/**
	 * The data types pane with the labels pane
	 */
	private JPanel _dataPaneWithLabels;
	/**
	 * Label for MySQL platform
	 */
	private JLabel _MySQLLabel;
	/**
	 * Label for Oracle platform
	 */
	private JLabel _OracleLabel;
	/**
	 * Label for DB2 platform
	 */
	private JLabel _DB2Label;
	/**
	 * Label for XML platform
	 */
	private JLabel _XMLLabel;
	/**
	 * Label for User Defined platform
	 */
	private JLabel _UserDefinedLabel;
	/**
	 * The data type controller
	 */
	private DataTypeController _controller;
	/**
	 * A list of the data type panes
	 */
	private ArrayList _dataTypePanes;
	/**
	 * The data types scroll pane
	 */
	private JScrollPane _dataScrollPane;
	
	/**
	 * Default constructor for DataTypesUI.
	 * Sets pane options and creates the GUI details
	 * 
	 * @param inCont The data type controller associated with the current sketch
	 */
	public DatatypesUI(DataTypeController inCont){
		_me = this;
		_controller = inCont;
		
		//Setup the frame
		this.setTitle("Data Type Information");
		this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		
		setSize();
		
		//Setup escape key to do same action as close button
		this.getRootPane().registerKeyboardAction( 
				new ActionListener(){
					public void actionPerformed(ActionEvent e){
						closeWindow();
					}
				},
				KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), 
				JComponent.WHEN_IN_FOCUSED_WINDOW);
		
		//Listen for window close event
		this.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent evt) {
                closeWindow();
            }
        });
		
		createDialogBox();
		
		this.setVisible(true);
	}
	
	/**
	 * Sets the size of the window
	 */
	public void setSize(){
		int numTypes = 0;
		if(_controller.is_useMySQL()){
			numTypes++;
		}
		if(_controller.is_useOracle()){
			numTypes++;
		}	
		if(_controller.is_useDB2()){
			numTypes++;
		}	
		if(_controller.is_useXML()){
			numTypes++;
		}	
		if(_controller.is_useUserDefined()){
			numTypes++;
		}

		int windowLength = BASE_WINDOW_LENGTH + (TYPE_WINDOW_LENGTH * numTypes);
		int windowHeight = BASE_WINDOW_HEIGHT + (TYPE_WINDOW_HEIGHT * _controller.getDataTypes().size());
		if(windowLength > MAX_WINDOW_LENGTH){
			windowLength = MAX_WINDOW_LENGTH;
		}
		if(windowHeight > MAX_WINDOW_HEIGHT){
			windowHeight = MAX_WINDOW_HEIGHT;
		}
		
		this.setSize(windowLength, windowHeight);
		this.setBounds(WINDOW_POS_X, WINDOW_POS_Y, windowLength, windowHeight);
	}
	
	/**
	 * Creates the dialog box elements and adds then to the dialog box
	 */
	public void createDialogBox(){
		_mainPane = new JPanel(new BorderLayout());
		_dataTypePanes = new ArrayList();
		
		//Create Panes
		createLabelPane();
		createDataPane();
		createButtonPane();
		
		//Add data pane to a scroll pane
		_dataPaneWithLabels = new JPanel(new BorderLayout());
		_dataPaneWithLabels.add(_labelAdjusterPane, BorderLayout.NORTH);
		_dataPaneWithLabels.add(_dataPane, BorderLayout.CENTER);
		_dataScrollPane = new JScrollPane(_dataPaneWithLabels, 
									JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
									JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		
		//Add panes to main pane
		_mainPane.add(_buttonPane, BorderLayout.SOUTH);
		_mainPane.add(_dataScrollPane, BorderLayout.CENTER);

		this.setContentPane(_mainPane);
	}
	
	/**
	 * Makes the button pane
	 */
	public void createButtonPane(){
		_buttonPane = new JPanel();
		_buttonPane.setLayout(new GridLayout(1,0));
		
		//Create Buttons and add them to the button pane
		JButton addButton = new JButton("Add data type");
		_buttonPane.add(addButton);
		JButton editAllButton = new JButton("Edit all");
		_buttonPane.add(editAllButton);
		JButton saveAllButton = new JButton("Save all");
		_buttonPane.add(saveAllButton);
		JButton resetButton = new JButton("Reset to defaults");
		_buttonPane.add(resetButton);
		JButton closeButton = new JButton("Close");
		_buttonPane.add(closeButton);
		
		//Create action listeners for buttons
		addButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				DataType newType = _controller.addNewDataType();
				DataTypePane curPane = new DataTypePane(newType, _controller, (DatatypesUI)_me);
				_dataTypePanes.add(curPane);
				_dataPane.add(curPane);
				_dataScrollPane.validate();
				_dataScrollPane.getVerticalScrollBar().setValue(_dataScrollPane.getVerticalScrollBar().getMaximum());
				curPane.setEditability(true);
			}
		});
		editAllButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				int numPanes = _dataTypePanes.size();
				for(int i=0; i<numPanes; i++){
					((DataTypePane)_dataTypePanes.get(i)).setEditability(true);
				}
			}
		});
		saveAllButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				int numTypes = _dataTypePanes.size();
				for(int i=0; i<numTypes; i++){
					((DataTypePane)_dataTypePanes.get(i)).setEditability(false);
				}
			}
		});
		resetButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				int op = JOptionPane.showConfirmDialog(Easik.getInstance().getFrame(), "Resetting the data types will set all atttributes to the default data type.\nAre you sure you want to continue?", "Confirm Deletion", JOptionPane.YES_NO_OPTION);
				if(op == JOptionPane.YES_OPTION){
					_controller.resetDataTypes();
					_dataTypePanes = new ArrayList();
					//Recreate Dialog with reset information
					createDialogBox();
					//Refresh view
					_me.validate();
				}
			}
		});
		closeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				closeWindow();
			}
		});
	}
	
	/**
	 * Create the data pane
	 */
	public void createDataPane(){
		_dataPane = new JPanel();
		
		_dataPane.setLayout(new BoxLayout(_dataPane, BoxLayout.PAGE_AXIS));
		
		//Create a DataTypePane for each data type and add it to the data pane
		int size = _controller.getDataTypes().size();
		for(int i=0; i<size; i++){
			DataTypePane curPane = new DataTypePane((DataType)_controller.getDataTypes().get(i), _controller, this);
			_dataTypePanes.add(curPane);
			_dataPane.add(curPane);
		}
	}
	
	/**
	 * Makes the label Pane
	 */
	public void createLabelPane(){
		_labelPane = new JPanel();
		_labelAdjusterPane = new JPanel();
		
		_labelAdjusterPane.setLayout(new BorderLayout());
		_labelPane.setLayout(new GridLayout(1,0));
		
		//Create spacer
		JButton spacerButton = new JButton("AAAAAA");
		//spacerButton.setVisible(false);
		_labelAdjusterPane.add(new JLabel("             "), BorderLayout.WEST);
		_labelAdjusterPane.add(_labelPane, BorderLayout.CENTER);
		
		//Create and add labels to label pane
		JLabel typeLabel = new JLabel(" Type Info ", JLabel.CENTER);
		_labelPane.add(typeLabel);
		if(_controller.is_useMySQL()){
			_labelPane.add(_MySQLLabel = new JLabel(" MySQL Type ", JLabel.CENTER));
		}
		if(_controller.is_useOracle()){
			_labelPane.add(_OracleLabel = new JLabel(" Oracle Type ", JLabel.CENTER));
		}	
		if(_controller.is_useDB2()){
			_labelPane.add(_DB2Label = new JLabel(" DB2 Type ", JLabel.CENTER));
		}	
		if(_controller.is_useXML()){
			_labelPane.add(_XMLLabel = new JLabel(" XML Type ", JLabel.CENTER));
		}	
		if(_controller.is_useUserDefined()){
			_labelPane.add(_UserDefinedLabel = new JLabel(" User Defined Type ", JLabel.CENTER));
		}	
	}
	
	/**
	 * Removes the data type pane from the GUI
	 * 
	 * @param inPane The data type pane to be removed
	 */
	public void removeDataTypePane(DataTypePane inPane){
		_dataPane.remove(inPane);
		_dataTypePanes.remove(inPane);
		this.validate();
	}
	
	/**
	 * Closes the window
	 */
	public void closeWindow(){
		//Check if all the data types have been saved
		boolean allSaved = true;
		int numTypes = _dataTypePanes.size();
		for(int i=0; i<numTypes; i++){
			allSaved = allSaved && !((DataTypePane)_dataTypePanes.get(i)).isEditable();
		}
		
		if(allSaved){
			this.dispose();
		}
		else{
			JOptionPane.showMessageDialog(
					this,
					"You have unsaved data types. Please save them and try again.",
					"Error",
					JOptionPane.ERROR_MESSAGE);
		}
	}
}
